# DonyayeGame API Documentation

## Overview
The DonyayeGame plugin provides a REST API service for external applications (like your Windows program) to interact with the system. All API requests require authentication using a 64-character API token.

## Authentication
All API requests must include the `X-API-Token` header with your API token:
```
X-API-Token: your_64_character_api_token_here
```

## Base URL
```
https://yourdomain.com/wp-json/donyaye-game/v1/
```

## Endpoints

### 1. Customer Registration
**POST** `/customer/register`

Register a new customer in the system.

**Request Body:**
```json
{
    "mobile": "09123456789",
    "first_name": "علی",
    "last_name": "احمدی",
    "national_code": "1234567890",
    "birth_date": "1990-01-01",
    "initial_balance": 10000
}
```

**Response (201 Created):**
```json
{
    "success": true,
    "message": "Customer registered successfully",
    "user_id": 123,
    "mobile": "09123456789",
    "balance": 10000
}
```

**Response (409 Conflict - User exists):**
```json
{
    "success": false,
    "message": "User already exists with this mobile number",
    "user_id": 123
}
```

### 2. Transaction Synchronization
**POST** `/transaction/sync`

Sync transactions between local (Windows) and online systems.

**Request Body:**
```json
{
    "type": "local",
    "user_mobile": "09123456789",
    "amount": 50000,
    "description": "Card payment at location",
    "reference_id": "LOCAL_TXN_001",
    "transaction_date": "2024-01-15T14:30:00"
}
```

**Response (201 Created):**
```json
{
    "success": true,
    "message": "Transaction synced successfully",
    "transaction_id": 456,
    "user_id": 123,
    "amount": 50000,
    "type": "local"
}
```

### 3. Get All Users
**GET** `/users`

Retrieve all users with pagination and filtering options.

**Query Parameters:**
- `per_page` (optional): Number of users per page (1-100, default: 50)
- `page` (optional): Page number (default: 1)
- `updated_after` (optional): Filter users updated after this date (ISO 8601 format)

**Example Request:**
```
GET /users?per_page=20&page=1&updated_after=2024-01-01T00:00:00
```

**Response (200 OK):**
```json
{
    "success": true,
    "users": [
        {
            "id": 123,
            "mobile": "09123456789",
            "first_name": "علی",
            "last_name": "احمدی",
            "national_code": "1234567890",
            "birth_date": "1990-01-01",
            "email": "user@example.com",
            "registration_date": "2024-01-01 10:00:00",
            "last_login": "2024-01-15 14:30:00",
            "last_update": "2024-01-15 14:30:00",
            "wallet_balance": 150000,
            "status": 0
        }
    ],
    "pagination": {
        "current_page": 1,
        "per_page": 20,
        "total_users": 150,
        "total_pages": 8
    }
}
```

### 4. Get User by Mobile
**GET** `/user/{mobile}`

Retrieve a specific user by mobile number.

**Example Request:**
```
GET /user/09123456789
```

**Response (200 OK):**
```json
{
    "success": true,
    "user": {
        "id": 123,
        "mobile": "09123456789",
        "first_name": "علی",
        "last_name": "احمدی",
        "national_code": "1234567890",
        "birth_date": "1990-01-01",
        "email": "user@example.com",
        "registration_date": "2024-01-01 10:00:00",
        "last_login": "2024-01-15 14:30:00",
        "last_update": "2024-01-15 14:30:00",
        "wallet_balance": 150000,
        "status": 0
    }
}
```

**Response (404 Not Found):**
```json
{
    "success": false,
    "message": "User not found with this mobile number"
}
```

### 5. Update User
**PUT** `/user/{mobile}`

Update user information.

**Request Body:**
```json
{
    "first_name": "علی",
    "last_name": "محمدی",
    "national_code": "1234567890",
    "birth_date": "1990-01-01",
    "email": "newemail@example.com"
}
```

**Response (200 OK):**
```json
{
    "success": true,
    "message": "User updated successfully",
    "user_id": 123
}
```

### **6. Get All Game Halls**
**GET** `/halls`

Retrieve all game halls with pagination and filtering options.

**Query Parameters:**
- `per_page` (optional): Number of halls per page (1-100, default: 50)
- `page` (optional): Page number (default: 1)
- `is_active` (optional): Filter by active status (0 or 1)

**Example Request:**
```
GET /halls?per_page=20&page=1&is_active=1
```

**Response (200 OK):**
```json
{
    "success": true,
    "halls": [
        {
            "id": 1,
            "name": "Game Hall A",
            "description": "Main gaming area",
            "images": "hall_a.jpg,hall_a2.jpg",
            "fee_type": 1,
            "fee_amount": 50000,
            "capacity_per_person": 1,
            "is_active": 1,
            "online_reservation_enabled": 1,
            "created_date": "2024-01-01T00:00:00",
            "modified_date": "2024-01-15T10:00:00"
        }
    ],
    "pagination": {
        "current_page": 1,
        "per_page": 20,
        "total_halls": 5,
        "total_pages": 1
    }
}
```

### **7. Get Game Hall by ID**
**GET** `/hall/{id}`

Retrieve a specific game hall by ID.

**Example Request:**
```
GET /hall/1
```

**Response (200 OK):**
```json
{
    "success": true,
    "hall": {
        "id": 1,
        "name": "Game Hall A",
        "description": "Main gaming area",
        "images": "hall_a.jpg,hall_a2.jpg",
        "fee_type": 1,
        "fee_amount": 50000,
        "capacity_per_person": 1,
        "is_active": 1,
        "online_reservation_enabled": 1,
        "created_date": "2024-01-01T00:00:00",
        "modified_date": "2024-01-15T10:00:00"
    }
}
```

### **8. Sync Game Hall**
**POST** `/hall/sync`

Sync game hall information between local (Windows) and online systems.

**Request Body:**
```json
{
    "id": 1,
    "name": "Game Hall A",
    "description": "Main gaming area",
    "images": "hall_a.jpg,hall_a2.jpg",
    "fee_type": 1,
    "fee_amount": 50000,
    "capacity_per_person": 1,
    "is_active": 1,
    "online_reservation_enabled": 1,
    "created_date": "2024-01-01T00:00:00",
    "modified_date": "2024-01-15T10:00:00"
}
```

**Response (200 OK):**
```json
{
    "success": true,
    "message": "Game hall synced successfully",
    "hall_id": 1
}
```

### **9. Get All Reservations**
**GET** `/reservations`

Retrieve all reservations with pagination and filtering options.

**Query Parameters:**
- `per_page` (optional): Number of reservations per page (1-100, default: 50)
- `page` (optional): Page number (default: 1)
- `customer_id` (optional): Filter by customer ID
- `hall_id` (optional): Filter by hall ID
- `reservation_date` (optional): Filter by reservation date (YYYY-MM-DD)
- `status` (optional): Filter by reservation status

**Example Request:**
```
GET /reservations?per_page=20&page=1&customer_id=1001&hall_id=1
```

**Response (200 OK):**
```json
{
    "success": true,
    "reservations": [
        {
            "id": 1,
            "hall_id": 1,
            "customer_id": 1001,
            "reservation_date": "2024-01-20",
            "reservation_time": "14:00:00",
            "approximate_arrival_time": "13:45:00",
            "paid_fee": 50000,
            "remaining_fee": 0,
            "is_active": 1,
            "status": 1,
            "created_date": "2024-01-15T10:00:00",
            "modified_date": "2024-01-15T10:00:00"
        }
    ],
    "pagination": {
        "current_page": 1,
        "per_page": 20,
        "total_reservations": 10,
        "total_pages": 1
    }
}
```

### **10. Sync Reservation**
**POST** `/reservation/sync`

Sync reservation information between local (Windows) and online systems.

**Request Body:**
```json
{
    "id": 1,
    "hall_id": 1,
    "customer_id": 1001,
    "reservation_date": "2024-01-20",
    "reservation_time": "14:00:00",
    "approximate_arrival_time": "13:45:00",
    "paid_fee": 50000,
    "remaining_fee": 0,
    "is_active": 1,
    "status": 1,
    "created_date": "2024-01-15T10:00:00",
    "modified_date": "2024-01-15T10:00:00"
}
```

**Response (200 OK):**
```json
{
    "success": true,
    "message": "Reservation synced successfully",
    "reservation_id": 1
}
```

## Error Responses

All endpoints return consistent error responses:

**400 Bad Request:**
```json
{
    "success": false,
    "message": "Validation error message"
}
```

**401 Unauthorized:**
```json
{
    "success": false,
    "message": "Invalid or missing API token"
}
```

**404 Not Found:**
```json
{
    "success": false,
    "message": "Resource not found"
}
```

**500 Internal Server Error:**
```json
{
    "success": false,
    "message": "Server error message"
}
```

## Data Validation

### Mobile Number
- Must be valid Iranian mobile format: `09xxxxxxxxx`
- Exactly 11 digits starting with `09`

### National Code
- Must be exactly 10 digits
- Format: `xxxxxxxxxx`

### Dates
- Birth date: `YYYY-MM-DD` format
- Transaction date: ISO 8601 format (`YYYY-MM-DDTHH:MM:SS`)

### Amounts
- Must be positive numbers
- No decimal places (integers only)

## Rate Limiting
Currently, no rate limiting is implemented. However, it's recommended to:
- Limit requests to reasonable frequency
- Implement proper error handling and retry logic
- Cache user data when possible

## Security Considerations
- Keep your API token secure and don't share it
- Use HTTPS for all API communications
- Regularly regenerate your API token
- Monitor API usage for suspicious activity

## Example Usage (C#)

```csharp
using System;
using System.Net.Http;
using System.Text;
using System.Threading.Tasks;
using Newtonsoft.Json;

public class DonyayeGameAPI
{
    private readonly HttpClient _client;
    private readonly string _baseUrl;
    private readonly string _apiToken;

    public DonyayeGameAPI(string baseUrl, string apiToken)
    {
        _baseUrl = baseUrl;
        _apiToken = apiToken;
        _client = new HttpClient();
        _client.DefaultRequestHeaders.Add("X-API-Token", _apiToken);
    }

    public async Task<bool> RegisterCustomer(CustomerData customer)
    {
        var json = JsonConvert.SerializeObject(customer);
        var content = new StringContent(json, Encoding.UTF8, "application/json");
        
        var response = await _client.PostAsync($"{_baseUrl}customer/register", content);
        var responseContent = await response.Content.ReadAsStringAsync();
        
        if (response.IsSuccessStatusCode)
        {
            var result = JsonConvert.DeserializeObject<ApiResponse>(responseContent);
            return result.Success;
        }
        
        return false;
    }

    public async Task<bool> SyncTransaction(TransactionData transaction)
    {
        var json = JsonConvert.SerializeObject(transaction);
        var content = new StringContent(json, Encoding.UTF8, "application/json");
        
        var response = await _client.PostAsync($"{_baseUrl}transaction/sync", content);
        var responseContent = await response.Content.ReadAsStringAsync();
        
        if (response.IsSuccessStatusCode)
        {
            var result = JsonConvert.DeserializeObject<ApiResponse>(responseContent);
            return result.Success;
        }
        
        return false;
    }
}

public class CustomerData
{
    public string Mobile { get; set; }
    public string FirstName { get; set; }
    public string LastName { get; set; }
    public string NationalCode { get; set; }
    public string BirthDate { get; set; }
    public decimal InitialBalance { get; set; }
}

public class TransactionData
{
    public string Type { get; set; }
    public string UserMobile { get; set; }
    public decimal Amount { get; set; }
    public string Description { get; set; }
    public string ReferenceId { get; set; }
    public DateTime? TransactionDate { get; set; }
}

public class ApiResponse
{
    public bool Success { get; set; }
    public string Message { get; set; }
}
```

## Support
For API support and questions, please contact your system administrator or refer to the plugin documentation. 