<?php
/**
 * Plugin Name: DonyayeGame
 * Plugin URI: https://unitweb.ir
 * Description: Complete DonyayeGame system with mobile OTP authentication, user dashboard, wallet management, and reservation system
 * Version: 1.0.0
 * Author: UnitWeb
 * Text Domain: donyaye-game
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * Network: false
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('DGA_PLUGIN_URL', plugin_dir_url(__FILE__));
define('DGA_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('DGA_PLUGIN_VERSION', '1.0.0');

// Include required files
require_once DGA_PLUGIN_PATH . 'includes/class-dga-auth.php';
require_once DGA_PLUGIN_PATH . 'includes/class-dga-sms.php';
require_once DGA_PLUGIN_PATH . 'includes/class-dga-forms.php';
require_once DGA_PLUGIN_PATH . 'includes/class-dga-ajax.php';
require_once DGA_PLUGIN_PATH . 'includes/class-dga-dashboard.php';
require_once DGA_PLUGIN_PATH . 'includes/class-dga-wallet.php';
require_once DGA_PLUGIN_PATH . 'includes/class-dga-reservations.php';
require_once DGA_PLUGIN_PATH . 'includes/class-dga-api-service.php';

// Initialize the plugin
function dga_init() {
    // Load text domain for translations
    load_plugin_textdomain('donyaye-game', false, dirname(plugin_basename(__FILE__)) . '/languages');
    
    // Initialize main authentication class
    new DGA_Auth();
    
    // Initialize SMS handler
    new DGA_SMS();
    
    // Initialize forms
    new DGA_Forms();
    
    // Initialize AJAX handlers
    new DGA_AJAX();
    
    // Initialize dashboard
    new DGA_Dashboard();
    
    // Initialize wallet system
    new DGA_Wallet();
    
    // Initialize reservations system
    new DGA_Reservations();
    
    // Initialize API service
    new DGA_API_Service();
}

// Add admin AJAX handlers
add_action('wp_ajax_dga_test_sms_config', 'dga_test_sms_config');
add_action('wp_ajax_dga_regenerate_api_token', 'dga_regenerate_api_token');

function dga_test_sms_config() {
    if (!wp_verify_nonce($_POST['nonce'], 'dga_test_sms')) {
        wp_die(json_encode(array('success' => false, 'data' => 'Security check failed')));
    }
    
    if (!current_user_can('manage_options')) {
        wp_die(json_encode(array('success' => false, 'data' => 'Insufficient permissions')));
    }
    
    $sms_handler = new DGA_SMS();
    $result = $sms_handler->test_sms_config();
    
    if (is_wp_error($result)) {
        wp_die(json_encode(array('success' => false, 'data' => $result->get_error_message())));
    }
    
    wp_die(json_encode(array('success' => true, 'data' => $result)));
}

function dga_regenerate_api_token() {
    if (!wp_verify_nonce($_POST['nonce'], 'dga_regenerate_token')) {
        wp_die(json_encode(array('success' => false, 'data' => 'Security check failed')));
    }
    
    if (!current_user_can('manage_options')) {
        wp_die(json_encode(array('success' => false, 'data' => 'Insufficient permissions')));
    }
    
    $api_service = new DGA_API_Service();
    $new_token = $api_service->generate_api_token();
    
    wp_die(json_encode(array('success' => true, 'data' => array('token' => $new_token))));
}
add_action('plugins_loaded', 'dga_init');

// Activation hook
register_activation_hook(__FILE__, 'dga_activate');
function dga_activate() {
    // Create custom tables
    global $wpdb;
    
    // OTP codes table
    $otp_table = $wpdb->prefix . 'dga_otp_codes';
    $charset_collate = $wpdb->get_charset_collate();
    
    $otp_sql = "CREATE TABLE $otp_table (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        mobile varchar(15) NOT NULL,
        otp_code varchar(6) NOT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        expires_at datetime NOT NULL,
        used tinyint(1) DEFAULT 0,
        PRIMARY KEY (id),
        KEY mobile (mobile),
        KEY otp_code (otp_code)
    ) $charset_collate;";
    
    // Wallet transactions table
    $wallet_table = $wpdb->prefix . 'dga_wallet_transactions';
    
    $wallet_sql = "CREATE TABLE $wallet_table (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        user_id bigint(20) NOT NULL,
        amount decimal(10,2) NOT NULL,
        type enum('credit', 'debit') NOT NULL,
        description text,
        reference_id varchar(100),
        status enum('pending', 'completed', 'failed') DEFAULT 'pending',
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY user_id (user_id),
        KEY status (status),
        KEY created_at (created_at)
    ) $charset_collate;";
    
    // Reservations table
    $reservations_table = $wpdb->prefix . 'dga_reservations';
    
    $reservations_sql = "CREATE TABLE $reservations_table (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        user_id bigint(20) NOT NULL,
        service_id varchar(100) NOT NULL,
        service_name varchar(255) NOT NULL,
        reservation_date date NOT NULL,
        reservation_time time NOT NULL,
        duration int(11) DEFAULT 60,
        status enum('pending', 'confirmed', 'cancelled', 'completed') DEFAULT 'pending',
        notes text,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY user_id (user_id),
        KEY service_id (service_id),
        KEY reservation_date (reservation_date),
        KEY status (status)
    ) $charset_collate;";
    
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($otp_sql);
    dbDelta($wallet_sql);
    dbDelta($reservations_sql);
    
    // Add customer role if it doesn't exist
    add_role('customer', 'Customer', array(
        'read' => true,
        'edit_posts' => false,
        'delete_posts' => false
    ));
    
    // Create dashboard page if it doesn't exist
    $dashboard_page = get_page_by_path('dashboard');
    if (!$dashboard_page) {
        wp_insert_post(array(
            'post_title' => 'Dashboard',
            'post_name' => 'dashboard',
            'post_status' => 'publish',
            'post_type' => 'page',
            'post_content' => '[dga_dashboard]'
        ));
    }
}

// Deactivation hook
register_deactivation_hook(__FILE__, 'dga_deactivate');
function dga_deactivate() {
    // Clean up if needed
}

// Enqueue scripts and styles
function dga_enqueue_scripts() {
    wp_enqueue_script('dga-main', DGA_PLUGIN_URL . 'assets/js/dga-main.js', array('jquery'), DGA_PLUGIN_VERSION, true);
    wp_enqueue_style('dga-main', DGA_PLUGIN_URL . 'assets/css/dga-main.css', array(), DGA_PLUGIN_VERSION);
    
    // Load RTL styles for Persian language
    if (get_locale() === 'fa_IR' || is_rtl()) {
        wp_enqueue_style('dga-main-rtl', DGA_PLUGIN_URL . 'assets/css/dga-main-rtl.css', array('dga-main'), DGA_PLUGIN_VERSION);
    }
    
    // Localize script for AJAX
    wp_localize_script('dga-main', 'dga_ajax', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('dga_nonce'),
        'dashboard_url' => home_url('/dashboard'),
        'strings' => array(
            'sending_otp' => __('Sending OTP...', 'donyaye-game'),
            'verifying_otp' => __('Verifying OTP...', 'donyaye-game'),
            'registering' => __('Registering...', 'donyaye-game'),
            'loading' => __('Loading...', 'donyaye-game'),
            'error' => __('An error occurred. Please try again.', 'donyaye-game')
        )
    ));
}
add_action('wp_enqueue_scripts', 'dga_enqueue_scripts');

// Add settings menu
function dga_admin_menu() {
    add_options_page(
        'DonyayeGame Settings',
        'DonyayeGame',
        'manage_options',
        'donyaye-game',
        'dga_settings_page'
    );
}
add_action('admin_menu', 'dga_admin_menu');

// Settings page
function dga_settings_page() {
            if (isset($_POST['submit'])) {
            update_option('dga_sms_api_key', sanitize_text_field($_POST['sms_api_key']));
            update_option('dga_sms_template_id', sanitize_text_field($_POST['sms_template_id']));
            update_option('dga_otp_expiry', intval($_POST['otp_expiry']));
            update_option('dga_wallet_currency', sanitize_text_field($_POST['wallet_currency']));
            update_option('dga_sms_debug', isset($_POST['sms_debug']) ? 1 : 0);
            echo '<div class="notice notice-success"><p>Settings saved successfully!</p></div>';
        }
        
        $sms_api_key = get_option('dga_sms_api_key', '');
        $sms_template_id = get_option('dga_sms_template_id', '');
        $otp_expiry = get_option('dga_otp_expiry', 300);
        $wallet_currency = get_option('dga_wallet_currency', 'IRR');
        $sms_debug = get_option('dga_sms_debug', 0);
    
    ?>
    <div class="wrap">
        <h1>DonyayeGame Settings</h1>
        <form method="post">
                            <h2>SMS Settings</h2>
                <div class="notice notice-info">
                    <p><strong>Important:</strong> SMS.ir has updated their API to v1. Make sure you're using the new <code>x-sms-ir-secure-token</code> from your SMS.ir dashboard, not the old API key.</p>
                </div>
                <table class="form-table">
                                        <tr>
                            <th scope="row">SMS.ir API Key</th>
                            <td>
                                <input type="text" name="sms_api_key" value="<?php echo esc_attr($sms_api_key); ?>" class="regular-text" />
                                <p class="description">
                                    Get your API key from <a href="https://sms.ir/web-service/" target="_blank">SMS.ir Web Service</a> dashboard.<br>
                                    <strong>Note:</strong> This is your <code>x-sms-ir-secure-token</code> from the new SMS.ir API v1
                                </p>
                            </td>
                        </tr>
                                                <tr>
                            <th scope="row">SMS Template ID</th>
                            <td>
                                <input type="text" name="sms_template_id" value="<?php echo esc_attr($sms_template_id); ?>" class="regular-text" />
                                <p class="description">
                                    Leave empty to use simple SMS instead of template SMS.<br>
                                    <strong>Template SMS:</strong> Uses your pre-approved SMS template with parameters<br>
                                    <strong>Simple SMS:</strong> Sends "کد تایید شما: [OTP]" as fallback
                                </p>
                            </td>
                        </tr>
                                                <tr>
                            <th scope="row">OTP Expiry (seconds)</th>
                            <td><input type="number" name="otp_expiry" value="<?php echo esc_attr($otp_expiry); ?>" class="small-text" /></td>
                        </tr>
                        <tr>
                            <th scope="row">Test SMS</th>
                            <td>
                                <button type="button" id="test-sms-btn" class="button button-secondary">Test SMS Configuration</button>
                                <span id="test-sms-result"></span>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Debug Mode</th>
                            <td>
                                <label>
                                    <input type="checkbox" name="sms_debug" value="1" <?php checked($sms_debug, 1); ?> />
                                    Enable SMS debugging (logs to WordPress error log)
                                </label>
                                <p class="description">Check this to see detailed SMS API requests and responses in your error log</p>
                            </td>
                        </tr>
                    </table>
                    
                                    <h2>Wallet Settings</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">Currency</th>
                        <td><input type="text" name="wallet_currency" value="<?php echo esc_attr($wallet_currency); ?>" class="small-text" /></td>
                    </tr>
                </table>
                
                <h2>API Integration Settings</h2>
                <div class="notice notice-info">
                    <p><strong>API Token:</strong> Use this token in your Windows application to authenticate API requests. The token should be sent in the <code>X-API-Token</code> header.</p>
                </div>
                <table class="form-table">
                    <tr>
                        <th scope="row">API Token</th>
                        <td>
                            <input type="text" id="api-token" value="<?php echo esc_attr(get_option('dga_api_token', '')); ?>" class="regular-text" readonly />
                            <button type="button" id="regenerate-token" class="button button-secondary">Regenerate Token</button>
                            <p class="description">This token is used to authenticate API requests from external applications</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">API Base URL</th>
                        <td>
                            <code><?php echo esc_url(rest_url('donyaye-game/v1/')); ?></code>
                            <p class="description">Base URL for all API endpoints</p>
                        </td>
                    </tr>
                </table>
                    
                    <?php submit_button(); ?>
                    
                    <script>
                    jQuery(document).ready(function($) {
                        // SMS Test Button
                        $('#test-sms-btn').on('click', function() {
                            var btn = $(this);
                            var result = $('#test-sms-result');
                            
                            btn.prop('disabled', true).text('Testing...');
                            result.html('');
                            
                            $.ajax({
                                url: ajaxurl,
                                type: 'POST',
                                data: {
                                    action: 'dga_test_sms_config',
                                    nonce: '<?php echo wp_create_nonce('dga_test_sms'); ?>'
                                },
                                success: function(response) {
                                    if (response.success) {
                                        result.html('<span style="color: green;">✓ SMS configuration is working! Credit: ' + response.data + '</span>');
                                    } else {
                                        result.html('<span style="color: red;">✗ ' + response.data + '</span>');
                                    }
                                },
                                error: function() {
                                    result.html('<span style="color: red;">✗ Network error occurred</span>');
                                },
                                complete: function() {
                                    btn.prop('disabled', false).text('Test SMS Configuration');
                                }
                            });
                        });
                        
                        // Token Regeneration Button
                        $('#regenerate-token').on('click', function() {
                            var btn = $(this);
                            var tokenField = $('#api-token');
                            
                            if (confirm('Are you sure you want to regenerate the API token? This will invalidate the current token and any applications using it will need to be updated.')) {
                                btn.prop('disabled', true).text('Regenerating...');
                                
                                $.ajax({
                                    url: ajaxurl,
                                    type: 'POST',
                                    data: {
                                        action: 'dga_regenerate_api_token',
                                        nonce: '<?php echo wp_create_nonce('dga_regenerate_token'); ?>'
                                    },
                                    success: function(response) {
                                        if (response.success) {
                                            tokenField.val(response.data.token);
                                            alert('API token regenerated successfully!');
                                        } else {
                                            alert('Failed to regenerate token: ' + response.data);
                                        }
                                    },
                                    error: function() {
                                        alert('Network error occurred while regenerating token');
                                    },
                                    complete: function() {
                                        btn.prop('disabled', false).text('Regenerate Token');
                                    }
                                });
                            }
                        });
                    });
                    </script>
        </form>
    </div>
    <?php
} 