<?php
/**
 * AJAX Handler Class
 */
class DGA_AJAX {
    
    public function __construct() {
        add_action('wp_ajax_nopriv_dga_send_otp', array($this, 'send_otp'));
        add_action('wp_ajax_nopriv_dga_verify_otp', array($this, 'verify_otp'));
        add_action('wp_ajax_nopriv_dga_complete_registration', array($this, 'complete_registration'));
        add_action('wp_ajax_dga_get_user_info', array($this, 'get_user_info'));
        add_action('wp_ajax_dga_logout', array($this, 'logout'));
        add_action('wp_ajax_dga_update_profile', array($this, 'update_profile'));
        add_action('wp_ajax_dga_get_form_html', array($this, 'get_form_html'));
    }
    
    /**
     * Send OTP via SMS
     */
    public function send_otp() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'dga_nonce')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Security check failed.', 'donyaye-game')
            )));
        }
        
        $mobile = sanitize_text_field($_POST['mobile']);
        
        if (empty($mobile)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Mobile number is required.', 'donyaye-game')
            )));
        }
        
        // Validate mobile format
        $sms_handler = new DGA_SMS();
        $valid_mobile = $sms_handler->validate_mobile($mobile);
        
        if (!$valid_mobile) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Invalid mobile number format. Please use 09xxxxxxxxx format.', 'donyaye-game')
            )));
        }
        
        // Check if user already exists
        $auth_handler = new DGA_Auth();
        $existing_user = $auth_handler->get_user_by_mobile($valid_mobile);
        
        // Send OTP
        $result = $sms_handler->send_otp($valid_mobile);
        
        if (is_wp_error($result)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => $result->get_error_message()
            )));
        }
        
        wp_die(json_encode(array(
            'success' => true,
            'message' => __('OTP sent successfully.', 'donyaye-game'),
            'mobile' => $valid_mobile,
            'user_exists' => $existing_user ? true : false
        )));
    }
    
    /**
     * Verify OTP and login/register user
     */
    public function verify_otp() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'dga_nonce')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Security check failed.', 'donyaye-game')
            )));
        }
        
        $mobile = sanitize_text_field($_POST['mobile']);
        $otp = sanitize_text_field($_POST['otp']);
        
        if (empty($mobile) || empty($otp)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Mobile number and OTP are required.', 'donyaye-game')
            )));
        }
        
        // Verify OTP
        $auth_handler = new DGA_Auth();
        
        if (!$auth_handler->verify_otp($mobile, $otp)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Invalid or expired OTP.', 'donyaye-game')
            )));
        }
        
        // Check if user exists
        $user = $auth_handler->get_user_by_mobile($mobile);
        
        if ($user) {
            // User exists, log them in
            $this->login_user($user);
            
            wp_die(json_encode(array(
                'success' => true,
                'message' => __('Login successful!', 'donyaye-game'),
                'redirect' => home_url('/dashboard'),
                'user_exists' => true
            )));
        } else {
            // User doesn't exist, mark OTP as used and proceed to registration
            $auth_handler->mark_otp_used($mobile, $otp);
            
            wp_die(json_encode(array(
                'success' => true,
                'message' => __('OTP verified. Please complete registration.', 'donyaye-game'),
                'mobile' => $mobile,
                'user_exists' => false
            )));
        }
    }
    
    /**
     * Complete user registration
     */
    public function complete_registration() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'dga_nonce')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Security check failed.', 'donyaye-game')
            )));
        }
        
        $mobile = sanitize_text_field($_POST['mobile']);
        $first_name = sanitize_text_field($_POST['first_name']);
        $last_name = sanitize_text_field($_POST['last_name']);
        $national_code = sanitize_text_field($_POST['national_code']);
        $birth_date = sanitize_text_field($_POST['birth_date']);
        
        if (empty($mobile) || empty($first_name) || empty($last_name) || empty($national_code) || empty($birth_date)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('All fields are required.', 'donyaye-game')
            )));
        }
        
        // Validate national code (Iranian national code validation)
        if (!$this->validate_national_code($national_code)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Invalid national code.', 'donyaye-game')
            )));
        }
        
        // Create user account
        $auth_handler = new DGA_Auth();
        $user_data = array(
            'first_name' => $first_name,
            'last_name' => $last_name,
            'national_code' => $national_code,
            'birth_date' => $birth_date
        );
        
        $user = $auth_handler->create_user_account($mobile, $user_data);
        
        if (is_wp_error($user)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => $user->get_error_message()
            )));
        }
        
        // Log the user in
        $this->login_user($user);
        
        wp_die(json_encode(array(
            'success' => true,
            'message' => __('Registration completed successfully!', 'donyaye-game'),
            'redirect' => home_url('/dashboard')
        )));
    }
    
    /**
     * Update user profile
     */
    public function update_profile() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'dga_nonce')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Security check failed.', 'donyaye-game')
            )));
        }
        
        if (!is_user_logged_in()) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('User not logged in.', 'donyaye-game')
            )));
        }
        
        $user_id = get_current_user_id();
        $first_name = sanitize_text_field($_POST['first_name']);
        $last_name = sanitize_text_field($_POST['last_name']);
        $national_code = sanitize_text_field($_POST['national_code']);
        $birth_date = sanitize_text_field($_POST['birth_date']);
        $email = sanitize_email($_POST['email']);
        
        if (empty($first_name) || empty($last_name) || empty($national_code) || empty($birth_date) || empty($email)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('All fields are required.', 'donyaye-game')
            )));
        }
        
        // Validate national code
        if (!$this->validate_national_code($national_code)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Invalid national code.', 'donyaye-game')
            )));
        }
        
        // Validate email
        if (!is_email($email)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Invalid email address.', 'donyaye-game')
            )));
        }
        
        // Update user data
        $user_data = array(
            'ID' => $user_id,
            'first_name' => $first_name,
            'last_name' => $last_name,
            'user_email' => $email,
            'display_name' => trim($first_name . ' ' . $last_name)
        );
        
        $result = wp_update_user($user_data);
        
        if (is_wp_error($result)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => $result->get_error_message()
            )));
        }
        
        // Update custom meta
        update_user_meta($user_id, 'national_code', $national_code);
        update_user_meta($user_id, 'birth_date', $birth_date);
        
        wp_die(json_encode(array(
            'success' => true,
            'message' => __('Profile updated successfully!', 'donyaye-game')
        )));
    }
    
    /**
     * Get user information
     */
    public function get_user_info() {
        if (!is_user_logged_in()) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('User not logged in.', 'donyaye-game')
            )));
        }
        
        $user = wp_get_current_user();
        $mobile = get_user_meta($user->ID, 'mobile_number', true);
        $national_code = get_user_meta($user->ID, 'national_code', true);
        $birth_date = get_user_meta($user->ID, 'birth_date', true);
        $registration_date = get_user_meta($user->ID, 'registration_date', true);
        
        wp_die(json_encode(array(
            'success' => true,
            'user' => array(
                'id' => $user->ID,
                'display_name' => $user->display_name,
                'email' => $user->user_email,
                'mobile' => $mobile,
                'national_code' => $national_code,
                'birth_date' => $birth_date,
                'registration_date' => $registration_date
            )
        )));
    }
    
    /**
     * Logout user
     */
    public function logout() {
        wp_logout();
        
        wp_die(json_encode(array(
            'success' => true,
            'message' => __('Logged out successfully.', 'donyaye-game'),
            'redirect' => home_url()
        )));
    }
    
    /**
     * Get form HTML for modal
     */
    public function get_form_html() {
        if (!wp_verify_nonce($_POST['nonce'], 'dga_nonce')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Security check failed.', 'donyaye-game')
            )));
        }
        
        $form_type = sanitize_text_field($_POST['form_type']);
        $forms_handler = new DGA_Forms();
        
        $html = $forms_handler->get_form_html($form_type);
        
        wp_die(json_encode(array(
            'success' => true,
            'html' => $html
        )));
    }
    
    /**
     * Login user
     */
    private function login_user($user) {
        wp_set_current_user($user->ID);
        wp_set_auth_cookie($user->ID);
        
        // Update last login time
        update_user_meta($user->ID, 'last_login', current_time('mysql'));
    }
    
    /**
     * Validate Iranian national code
     */
    private function validate_national_code($code) {
        // Remove any non-digit characters
        $code = preg_replace('/[^0-9]/', '', $code);
        
        // Check length
        if (strlen($code) !== 10) {
            return false;
        }
        
        // Check if all digits are the same
        if (preg_match('/^(\d)\1{9}$/', $code)) {
            return false;
        }
        
        // Validate using algorithm
        $sum = 0;
        for ($i = 0; $i < 9; $i++) {
            $sum += intval($code[$i]) * (10 - $i);
        }
        
        $remainder = $sum % 11;
        $control_digit = intval($code[9]);
        
        if ($remainder < 2) {
            return $control_digit === $remainder;
        } else {
            return $control_digit === (11 - $remainder);
        }
    }
} 