<?php
/**
 * API Service Class for External Integration
 */
class DGA_API_Service {
    
    private $api_token;
    private $api_token_option = 'dga_api_token';
    
    public function __construct() {
        $this->api_token = get_option($this->api_token_option, '');
        
        // Generate token if it doesn't exist
        if (empty($this->api_token)) {
            $this->generate_api_token();
        }
        
        // Register REST API endpoints
        add_action('rest_api_init', array($this, 'register_rest_routes'));
    }
    
    /**
     * Generate a new 64-character API token
     */
    public function generate_api_token() {
        $token = bin2hex(random_bytes(32)); // 64 characters
        update_option($this->api_token_option, $token);
        $this->api_token = $token;
        return $token;
    }
    
    /**
     * Get current API token
     */
    public function get_api_token() {
        return $this->api_token;
    }
    
    /**
     * Validate API token
     */
    public function validate_token($token) {
        return $token === $this->api_token;
    }
    
    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        // Customer registration endpoint
        register_rest_route('donyaye-game/v1', '/customer/register', array(
            'methods' => 'POST',
            'callback' => array($this, 'register_customer'),
            'permission_callback' => array($this, 'check_api_permission'),
            'args' => array(
                'id' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'name' => array(
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'surname' => array(
                    'required' => true,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'gender' => array(
                    'required' => false,
                    'type' => 'integer',
                    'default' => 0
                ),
                'national_code' => array(
                    'required' => true,
                    'type' => 'string',
                    'validate_callback' => array($this, 'validate_national_code')
                ),
                'mobile_number' => array(
                    'required' => true,
                    'type' => 'string',
                    'validate_callback' => array($this, 'validate_mobile')
                ),
                'date_of_birth' => array(
                    'required' => true,
                    'type' => 'string',
                    'format' => 'date'
                ),
                'image' => array(
                    'required' => false,
                    'type' => 'string'
                ),
                'balance' => array(
                    'required' => false,
                    'type' => 'number',
                    'default' => 0
                ),
                'membership_date' => array(
                    'required' => false,
                    'type' => 'string',
                    'format' => 'date'
                ),
                'last_entry_date' => array(
                    'required' => false,
                    'type' => 'string',
                    'format' => 'date-time'
                ),
                'is_active' => array(
                    'required' => false,
                    'type' => 'integer',
                    'default' => 1
                ),
                'created_date' => array(
                    'required' => false,
                    'type' => 'string',
                    'format' => 'date-time'
                ),
                'modified_date' => array(
                    'required' => false,
                    'type' => 'string',
                    'format' => 'date-time'
                )
            )
        ));
        
        // Transaction sync endpoint
        register_rest_route('donyaye-game/v1', '/transaction/sync', array(
            'methods' => 'POST',
            'callback' => array($this, 'sync_transaction'),
            'permission_callback' => array($this, 'check_api_permission'),
            'args' => array(
                'id' => array(
                    'required' => false,
                    'type' => 'integer'
                ),
                'customer_id' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
                'amount' => array(
                    'required' => true,
                    'type' => 'number'
                ),
                'type' => array(
                    'required' => true,
                    'type' => 'integer'
                ),
                'previous_balance' => array(
                    'required' => false,
                    'type' => 'number'
                ),
                'new_balance' => array(
                    'required' => false,
                    'type' => 'number'
                ),
                'transaction_date' => array(
                    'required' => false,
                    'type' => 'string',
                    'format' => 'date-time'
                ),
                'description' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'card_number' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'transaction_number' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'track_no' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'resp_code' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'resp_message' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'terminal_number' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'reference_number' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'is_online_transaction' => array(
                    'required' => false,
                    'type' => 'integer',
                    'default' => 0
                ),
                'is_successful' => array(
                    'required' => false,
                    'type' => 'integer',
                    'default' => 1
                ),
                'created_by' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'created_date' => array(
                    'required' => false,
                    'type' => 'string',
                    'format' => 'date-time'
                ),
                'modified_date' => array(
                    'required' => false,
                    'type' => 'string',
                    'format' => 'date-time'
                )
            )
        ));
        
        // Get all users endpoint
        register_rest_route('donyaye-game/v1', '/users', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_all_users'),
            'permission_callback' => array($this, 'check_api_permission'),
            'args' => array(
                'per_page' => array(
                    'required' => false,
                    'type' => 'integer',
                    'minimum' => 1,
                    'maximum' => 100,
                    'default' => 50
                ),
                'page' => array(
                    'required' => false,
                    'type' => 'integer',
                    'minimum' => 1,
                    'default' => 1
                ),
                'updated_after' => array(
                    'required' => false,
                    'type' => 'string',
                    'format' => 'date-time'
                )
            )
        ));
        
        // Get user by mobile endpoint
        register_rest_route('donyaye-game/v1', '/user/(?P<mobile>[0-9]+)', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_user_by_mobile'),
            'permission_callback' => array($this, 'check_api_permission'),
            'args' => array(
                'mobile' => array(
                    'required' => true,
                    'type' => 'string',
                    'validate_callback' => array($this, 'validate_mobile')
                )
            )
        ));
        
        // Update user endpoint
        register_rest_route('donyaye-game/v1', '/user/(?P<mobile>[0-9]+)', array(
            'methods' => 'PUT',
            'callback' => array($this, 'update_user'),
            'permission_callback' => array($this, 'check_api_permission'),
            'args' => array(
                'mobile' => array(
                    'required' => true,
                    'type' => 'string',
                    'validate_callback' => array($this, 'validate_mobile')
                ),
                'first_name' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'last_name' => array(
                    'required' => false,
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field'
                ),
                'national_code' => array(
                    'required' => false,
                    'type' => 'string',
                    'validate_callback' => array($this, 'validate_national_code')
                ),
                'birth_date' => array(
                    'required' => false,
                    'type' => 'string',
                    'format' => 'date'
                ),
                'email' => array(
                    'required' => false,
                    'type' => 'string',
                    'format' => 'email'
                )
            )
        ));
    }
    
    /**
     * Check API permission using token
     */
    public function check_api_permission($request) {
        $token = $request->get_header('X-API-Token');
        if (empty($token)) {
            return false;
        }
        return $this->validate_token($token);
    }
    
    /**
     * Validate mobile number format
     */
    public function validate_mobile($mobile) {
        return preg_match('/^09[0-9]{9}$/', $mobile);
    }
    
    /**
     * Validate national code format
     */
    public function validate_national_code($code) {
        return preg_match('/^[0-9]{10}$/', $code);
    }
    
    /**
     * Register new customer
     */
    public function register_customer($request) {
        $id = $request->get_param('id');
        $name = $request->get_param('name');
        $surname = $request->get_param('surname');
        $gender = $request->get_param('gender');
        $national_code = $request->get_param('national_code');
        $mobile_number = $request->get_param('mobile_number');
        $date_of_birth = $request->get_param('date_of_birth');
        $image = $request->get_param('image');
        $balance = $request->get_param('balance');
        $membership_date = $request->get_param('membership_date');
        $last_entry_date = $request->get_param('last_entry_date');
        $is_active = $request->get_param('is_active');
        $created_date = $request->get_param('created_date');
        $modified_date = $request->get_param('modified_date');
        
        // Check if user already exists
        $existing_user = $this->get_user_by_mobile_number($mobile_number);
        if ($existing_user) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => 'User already exists with this mobile number',
                'user_id' => $existing_user->ID
            ), 409);
        }
        
        // Create user account
        $user_data = array(
            'user_login' => 'user_' . $mobile_number,
            'user_pass' => wp_generate_password(12, false),
            'user_email' => $mobile_number . '@donyaye-game.local',
            'first_name' => $name,
            'last_name' => $surname,
            'display_name' => trim($name . ' ' . $surname),
            'role' => 'customer'
        );
        
        $user_id = wp_insert_user($user_data);
        
        if (is_wp_error($user_id)) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => 'Failed to create user account: ' . $user_id->get_error_message()
            ), 500);
        }
        
        // Add custom user meta matching your Windows app structure
        update_user_meta($user_id, 'local_id', $id);
        update_user_meta($user_id, 'mobile_number', $mobile_number);
        update_user_meta($user_id, 'national_code', $national_code);
        update_user_meta($user_id, 'gender', $gender);
        update_user_meta($user_id, 'date_of_birth', $date_of_birth);
        update_user_meta($user_id, 'image', $image);
        update_user_meta($user_id, 'membership_date', $membership_date ?: current_time('mysql'));
        update_user_meta($user_id, 'last_entry_date', $last_entry_date);
        update_user_meta($user_id, 'is_active', $is_active);
        update_user_meta($user_id, 'registration_source', 'api');
        update_user_meta($user_id, 'local_created_date', $created_date);
        update_user_meta($user_id, 'local_modified_date', $modified_date);
        
        // Add initial balance if specified
        if ($balance > 0) {
            $wallet = new DGA_Wallet();
            $transaction_id = $wallet->create_transaction(
                $user_id,
                $balance,
                'credit',
                'Initial balance from API registration',
                'api_registration'
            );
            
            if ($transaction_id) {
                $wallet->update_transaction_status($transaction_id, 'completed');
            }
        }
        
        return new WP_REST_Response(array(
            'success' => true,
            'message' => 'Customer registered successfully',
            'user_id' => $user_id,
            'local_id' => $id,
            'mobile_number' => $mobile_number,
            'balance' => $balance
        ), 201);
    }
    
    /**
     * Sync transaction between local and online
     */
    public function sync_transaction($request) {
        $id = $request->get_param('id');
        $customer_id = $request->get_param('customer_id');
        $amount = $request->get_param('amount');
        $type = $request->get_param('type');
        $previous_balance = $request->get_param('previous_balance');
        $new_balance = $request->get_param('new_balance');
        $transaction_date = $request->get_param('transaction_date');
        $description = $request->get_param('description');
        $card_number = $request->get_param('card_number');
        $transaction_number = $request->get_param('transaction_number');
        $track_no = $request->get_param('track_no');
        $resp_code = $request->get_param('resp_code');
        $resp_message = $request->get_param('resp_message');
        $terminal_number = $request->get_param('terminal_number');
        $reference_number = $request->get_param('reference_number');
        $is_online_transaction = $request->get_param('is_online_transaction');
        $is_successful = $request->get_param('is_successful');
        $created_by = $request->get_param('created_by');
        $created_date = $request->get_param('created_date');
        $modified_date = $request->get_param('modified_date');
        
        // Get user by local ID or create a reference
        $user = $this->get_user_by_local_id($customer_id);
        if (!$user) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => 'User not found with this customer ID'
            ), 404);
        }
        
        $user_id = $user->ID;
        $wallet = new DGA_Wallet();
        
        // Determine transaction type for wallet
        $wallet_type = ($type == 1) ? 'credit' : 'debit'; // Assuming 1=credit, 0=debit
        
        // Create transaction record
        $transaction_id = $wallet->create_transaction(
            $user_id,
            $amount,
            $wallet_type,
            $description ?: "Transaction sync from local system",
            $reference_number ?: $transaction_number ?: uniqid("sync_")
        );
        
        if (!$transaction_id) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => 'Failed to create transaction record'
            ), 500);
        }
        
        // Update transaction with additional details
        global $wpdb;
        $table_name = $wpdb->prefix . 'dga_wallet_transactions';
        
        $update_data = array(
            'status' => $is_successful ? 'completed' : 'failed',
            'reference_id' => $id
        );
        
        // Add additional transaction details as meta
        $meta_data = array(
            'local_id' => $id,
            'previous_balance' => $previous_balance,
            'new_balance' => $new_balance,
            'card_number' => $card_number,
            'transaction_number' => $transaction_number,
            'track_no' => $track_no,
            'resp_code' => $resp_code,
            'resp_message' => $resp_message,
            'terminal_number' => $terminal_number,
            'is_online_transaction' => $is_online_transaction,
            'created_by' => $created_by,
            'local_created_date' => $created_date,
            'local_modified_date' => $modified_date
        );
        
        // Update transaction
        $wpdb->update(
            $table_name,
            $update_data,
            array('id' => $transaction_id),
            array('%s', '%s'),
            array('%d')
        );
        
        // Update transaction date if provided
        if ($transaction_date) {
            $wpdb->update(
                $table_name,
                array('created_at' => $transaction_date),
                array('id' => $transaction_id),
                array('%s'),
                array('%d')
            );
        }
        
        // Store additional meta data
        foreach ($meta_data as $key => $value) {
            if ($value !== null) {
                update_post_meta($transaction_id, 'dga_' . $key, $value);
            }
        }
        
        return new WP_REST_Response(array(
            'success' => true,
            'message' => 'Transaction synced successfully',
            'transaction_id' => $transaction_id,
            'user_id' => $user_id,
            'local_id' => $id,
            'amount' => $amount,
            'type' => $type
        ), 201);
    }
    
    /**
     * Get all users with pagination
     */
    public function get_all_users($request) {
        $per_page = $request->get_param('per_page');
        $page = $request->get_param('page');
        $updated_after = $request->get_param('updated_after');
        
        $args = array(
            'role' => 'customer',
            'number' => $per_page,
            'paged' => $page,
            'orderby' => 'registered',
            'order' => 'DESC'
        );
        
        // Filter by update date if specified
        if ($updated_after) {
            $args['meta_query'] = array(
                array(
                    'key' => 'last_update',
                    'value' => $updated_after,
                    'compare' => '>',
                    'type' => 'DATETIME'
                )
            );
        }
        
        $users = get_users($args);
        $total_users = count_users()['avail_roles']['customer'] ?? 0;
        
        $user_data = array();
        foreach ($users as $user) {
            $user_data[] = $this->format_user_data($user);
        }
        
        return new WP_REST_Response(array(
            'success' => true,
            'users' => $user_data,
            'pagination' => array(
                'current_page' => $page,
                'per_page' => $per_page,
                'total_users' => $total_users,
                'total_pages' => ceil($total_users / $per_page)
            )
        ), 200);
    }
    
    /**
     * Get user by mobile number
     */
    public function get_user_by_mobile($request) {
        $mobile = $request->get_param('mobile');
        
        $user = $this->get_user_by_mobile_number($mobile);
        if (!$user) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => 'User not found with this mobile number'
            ), 404);
        }
        
        return new WP_REST_Response(array(
            'success' => true,
            'user' => $this->format_user_data($user)
        ), 200);
    }
    
    /**
     * Update user information
     */
    public function update_user($request) {
        $mobile = $request->get_param('mobile');
        $first_name = $request->get_param('first_name');
        $last_name = $request->get_param('last_name');
        $national_code = $request->get_param('national_code');
        $birth_date = $request->get_param('birth_date');
        $email = $request->get_param('email');
        
        $user = $this->get_user_by_mobile_number($mobile);
        if (!$user) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => 'User not found with this mobile number'
            ), 404);
        }
        
        $user_id = $user->ID;
        $update_data = array('ID' => $user_id);
        
        // Update user fields if provided
        if ($first_name) {
            $update_data['first_name'] = $first_name;
        }
        if ($last_name) {
            $update_data['last_name'] = $last_name;
        }
        if ($email) {
            $update_data['user_email'] = $email;
        }
        if ($first_name || $last_name) {
            $update_data['display_name'] = trim(($first_name ?: get_user_meta($user_id, 'first_name', true)) . ' ' . ($last_name ?: get_user_meta($user_id, 'last_name', true)));
        }
        
        // Update user if there are changes
        if (count($update_data) > 1) {
            $result = wp_update_user($update_data);
            if (is_wp_error($result)) {
                return new WP_REST_Response(array(
                    'success' => false,
                    'message' => 'Failed to update user: ' . $result->get_error_message()
                ), 500);
            }
        }
        
        // Update user meta if provided
        if ($national_code) {
            update_user_meta($user_id, 'national_code', $national_code);
        }
        if ($birth_date) {
            update_user_meta($user_id, 'birth_date', $birth_date);
        }
        
        // Update last update timestamp
        update_user_meta($user_id, 'last_update', current_time('mysql'));
        
        return new WP_REST_Response(array(
            'success' => true,
            'message' => 'User updated successfully',
            'user_id' => $user_id
        ), 200);
    }
    
    /**
     * Get user by mobile number (helper method)
     */
    private function get_user_by_mobile_number($mobile) {
        $users = get_users(array(
            'meta_key' => 'mobile_number',
            'meta_value' => $mobile,
            'role' => 'customer',
            'number' => 1
        ));
        
        return !empty($users) ? $users[0] : false;
    }
    
    /**
     * Get user by local ID (helper method)
     */
    private function get_user_by_local_id($local_id) {
        $users = get_users(array(
            'meta_key' => 'local_id',
            'meta_value' => $local_id,
            'role' => 'customer',
            'number' => 1
        ));
        
        return !empty($users) ? $users[0] : false;
    }
    
    /**
     * Format user data for API response
     */
    private function format_user_data($user) {
        $wallet = new DGA_Wallet();
        $balance = $wallet->get_wallet_balance($user->ID);
        
        return array(
            'id' => $user->ID,
            'mobile' => get_user_meta($user->ID, 'mobile_number', true),
            'first_name' => get_user_meta($user->ID, 'first_name', true),
            'last_name' => get_user_meta($user->ID, 'last_name', true),
            'national_code' => get_user_meta($user->ID, 'national_code', true),
            'birth_date' => get_user_meta($user->ID, 'birth_date', true),
            'email' => $user->user_email,
            'registration_date' => get_user_meta($user->ID, 'registration_date', true),
            'last_login' => get_user_meta($user->ID, 'last_login', true),
            'last_update' => get_user_meta($user->ID, 'last_update', true),
            'wallet_balance' => $balance,
            'status' => $user->user_status
        );
    }
} 