<?php
/**
 * Dashboard Class
 */
class DGA_Dashboard {
    
    public function __construct() {
        add_shortcode('dga_dashboard', array($this, 'dashboard_shortcode'));
        add_action('wp_ajax_dga_load_dashboard_section', array($this, 'load_dashboard_section'));
        add_action('wp_ajax_nopriv_dga_load_dashboard_section', array($this, 'load_dashboard_section'));
    }
    
    /**
     * Dashboard shortcode
     */
    public function dashboard_shortcode($atts) {
        if (!is_user_logged_in()) {
            return '<p>' . __('Please login to access the dashboard.', 'donyaye-game') . '</p>';
        }
        
        $atts = shortcode_atts(array(
            'class' => 'dga-dashboard'
        ), $atts);
        
        ob_start();
        ?>
        <div class="<?php echo esc_attr($atts['class']); ?>">
            <div class="dga-dashboard-container">
                <!-- Sidebar -->
                <div class="dga-sidebar">
                    <div class="dga-user-info">
                        <?php
                        $user = wp_get_current_user();
                        $mobile = get_user_meta($user->ID, 'mobile_number', true);
                        ?>
                        <div class="user-avatar">
                            <span class="avatar-text"><?php echo strtoupper(substr($user->first_name, 0, 1) . substr($user->last_name, 0, 1)); ?></span>
                        </div>
                        <div class="user-details">
                            <h4><?php echo esc_html($user->display_name); ?></h4>
                            <p><?php echo esc_html($mobile); ?></p>
                        </div>
                    </div>
                    
                    <nav class="dga-nav">
                        <ul>
                            <li class="nav-item active" data-section="profile">
                                <a href="#profile">
                                    <i class="dga-icon profile-icon"></i>
                                    <span><?php _e('Edit Profile', 'donyaye-game'); ?></span>
                                </a>
                            </li>
                            <li class="nav-item" data-section="wallet">
                                <a href="#wallet">
                                    <i class="dga-icon wallet-icon"></i>
                                    <span><?php _e('Charge Wallet', 'donyaye-game'); ?></span>
                                </a>
                            </li>
                            <li class="nav-item" data-section="transactions">
                                <a href="#transactions">
                                    <i class="dga-icon transactions-icon"></i>
                                    <span><?php _e('Transaction History', 'donyaye-game'); ?></span>
                                </a>
                            </li>
                            <li class="nav-item" data-section="reservations">
                                <a href="#reservations">
                                    <i class="dga-icon reservations-icon"></i>
                                    <span><?php _e('Reservation History', 'donyaye-game'); ?></span>
                                </a>
                            </li>
                        </ul>
                    </nav>
                    
                    <div class="dga-sidebar-footer">
                        <a href="<?php echo wp_logout_url(); ?>" class="dga-logout-btn">
                            <i class="dga-icon logout-icon"></i>
                            <?php _e('Logout', 'donyaye-game'); ?>
                        </a>
                    </div>
                </div>
                
                <!-- Main Content -->
                <div class="dga-main-content">
                    <div class="dga-content-header">
                        <h2 id="section-title"><?php _e('Edit Profile', 'donyaye-game'); ?></h2>
                    </div>
                    
                    <div class="dga-content-body" id="dashboard-content">
                        <?php echo $this->get_profile_section(); ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Load dashboard section via AJAX
     */
    public function load_dashboard_section() {
        if (!wp_verify_nonce($_POST['nonce'], 'dga_nonce')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Security check failed.', 'donyaye-game')
            )));
        }
        
        $section = sanitize_text_field($_POST['section']);
        
        switch ($section) {
            case 'profile':
                $content = $this->get_profile_section();
                $title = __('Edit Profile', 'donyaye-game');
                break;
            case 'wallet':
                $content = $this->get_wallet_section();
                $title = __('Charge Wallet', 'donyaye-game');
                break;
            case 'transactions':
                $content = $this->get_transactions_section();
                $title = __('Transaction History', 'donyaye-game');
                break;
            case 'reservations':
                $content = $this->get_reservations_section();
                $title = __('Reservation History', 'donyaye-game');
                break;
            default:
                $content = '<p>' . __('Section not found.', 'donyaye-game') . '</p>';
                $title = __('Dashboard', 'donyaye-game');
        }
        
        wp_die(json_encode(array(
            'success' => true,
            'content' => $content,
            'title' => $title
        )));
    }
    
    /**
     * Get profile section content
     */
    private function get_profile_section() {
        $user = wp_get_current_user();
        $mobile = get_user_meta($user->ID, 'mobile_number', true);
        $national_code = get_user_meta($user->ID, 'national_code', true);
        $birth_date = get_user_meta($user->ID, 'birth_date', true);
        
        ob_start();
        ?>
        <div class="dga-profile-section">
            <form class="dga-profile-form" id="profile-form">
                <div class="form-row">
                    <div class="form-group">
                        <label for="first_name"><?php _e('First Name:', 'donyaye-game'); ?></label>
                        <input type="text" id="first_name" name="first_name" value="<?php echo esc_attr($user->first_name); ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="last_name"><?php _e('Last Name:', 'donyaye-game'); ?></label>
                        <input type="text" id="last_name" name="last_name" value="<?php echo esc_attr($user->last_name); ?>" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="mobile"><?php _e('Mobile Number:', 'donyaye-game'); ?></label>
                        <input type="tel" id="mobile" name="mobile" value="<?php echo esc_attr($mobile); ?>" readonly>
                        <small><?php _e('Mobile number cannot be changed', 'donyaye-game'); ?></small>
                    </div>
                    <div class="form-group">
                        <label for="national_code"><?php _e('National Code:', 'donyaye-game'); ?></label>
                        <input type="text" id="national_code" name="national_code" value="<?php echo esc_attr($national_code); ?>" maxlength="10" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="birth_date"><?php _e('Date of Birth:', 'donyaye-game'); ?></label>
                        <input type="date" id="birth_date" name="birth_date" value="<?php echo esc_attr($birth_date); ?>" required>
                    </div>
                    <div class="form-group">
                        <label for="email"><?php _e('Email:', 'donyaye-game'); ?></label>
                        <input type="email" id="email" name="email" value="<?php echo esc_attr($user->user_email); ?>" required>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="dga-btn dga-btn-primary">
                        <?php _e('Update Profile', 'donyaye-game'); ?>
                    </button>
                </div>
            </form>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get wallet section content
     */
    private function get_wallet_section() {
        $user = wp_get_current_user();
        $wallet_balance = $this->get_wallet_balance($user->ID);
        $currency = get_option('dga_wallet_currency', 'IRR');
        
        ob_start();
        ?>
        <div class="dga-wallet-section">
            <div class="wallet-balance">
                <h3><?php _e('Current Balance', 'donyaye-game'); ?></h3>
                <div class="balance-amount">
                    <span class="amount"><?php echo number_format($wallet_balance, 0); ?></span>
                    <span class="currency"><?php echo esc_html($currency); ?></span>
                </div>
            </div>
            
            <div class="charge-wallet">
                <h3><?php _e('Charge Wallet', 'donyaye-game'); ?></h3>
                <form class="dga-charge-form" id="charge-form">
                    <div class="form-group">
                        <label for="charge_amount"><?php _e('Amount:', 'donyaye-game'); ?></label>
                        <input type="number" id="charge_amount" name="amount" min="1000" step="1000" required>
                        <small><?php printf(__('Minimum amount: %s %s', 'donyaye-game'), '1,000', $currency); ?></small>
                    </div>
                    
                    <div class="form-group">
                        <label for="payment_method"><?php _e('Payment Method:', 'donyaye-game'); ?></label>
                        <select id="payment_method" name="payment_method" required>
                            <option value=""><?php _e('Select payment method', 'donyaye-game'); ?></option>
                            <option value="online"><?php _e('Online Payment', 'donyaye-game'); ?></option>
                            <option value="bank_transfer"><?php _e('Bank Transfer', 'donyaye-game'); ?></option>
                        </select>
                    </div>
                    
                    <div class="form-actions">
                        <button type="submit" class="dga-btn dga-btn-primary">
                            <?php _e('Charge Wallet', 'donyaye-game'); ?>
                        </button>
                    </div>
                </form>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get transactions section content
     */
    private function get_transactions_section() {
        $user = wp_get_current_user();
        $transactions = $this->get_user_transactions($user->ID);
        $currency = get_option('dga_wallet_currency', 'IRR');
        
        ob_start();
        ?>
        <div class="dga-transactions-section">
            <h3><?php _e('Transaction History', 'donyaye-game'); ?></h3>
            
            <?php if (empty($transactions)): ?>
                <div class="no-data">
                    <p><?php _e('No transactions found.', 'donyaye-game'); ?></p>
                </div>
            <?php else: ?>
                <div class="transactions-list">
                    <table class="dga-table">
                        <thead>
                            <tr>
                                <th><?php _e('Date', 'donyaye-game'); ?></th>
                                <th><?php _e('Type', 'donyaye-game'); ?></th>
                                <th><?php _e('Amount', 'donyaye-game'); ?></th>
                                <th><?php _e('Description', 'donyaye-game'); ?></th>
                                <th><?php _e('Status', 'donyaye-game'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($transactions as $transaction): ?>
                                <tr>
                                    <td><?php echo date('Y-m-d H:i', strtotime($transaction->created_at)); ?></td>
                                    <td>
                                        <span class="transaction-type <?php echo $transaction->type; ?>">
                                            <?php echo $transaction->type === 'credit' ? __('Credit', 'donyaye-game') : __('Debit', 'donyaye-game'); ?>
                                        </span>
                                    </td>
                                    <td class="amount">
                                        <?php echo number_format($transaction->amount, 0); ?> <?php echo esc_html($currency); ?>
                                    </td>
                                    <td><?php echo esc_html($transaction->description); ?></td>
                                    <td>
                                        <span class="status <?php echo $transaction->status; ?>">
                                            <?php echo ucfirst($transaction->status); ?>
                                        </span>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get reservations section content
     */
    private function get_reservations_section() {
        $user = wp_get_current_user();
        $reservations = $this->get_user_reservations($user->ID);
        
        ob_start();
        ?>
        <div class="dga-reservations-section">
            <h3><?php _e('Reservation History', 'donyaye-game'); ?></h3>
            
            <?php if (empty($reservations)): ?>
                <div class="no-data">
                    <p><?php _e('No reservations found.', 'donyaye-game'); ?></p>
                </div>
            <?php else: ?>
                <div class="reservations-list">
                    <table class="dga-table">
                        <thead>
                            <tr>
                                <th><?php _e('Service', 'donyaye-game'); ?></th>
                                <th><?php _e('Date', 'donyaye-game'); ?></th>
                                <th><?php _e('Time', 'donyaye-game'); ?></th>
                                <th><?php _e('Duration', 'donyaye-game'); ?></th>
                                <th><?php _e('Status', 'donyaye-game'); ?></th>
                                <th><?php _e('Actions', 'donyaye-game'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($reservations as $reservation): ?>
                                <tr>
                                    <td><?php echo esc_html($reservation->service_name); ?></td>
                                    <td><?php echo date('Y-m-d', strtotime($reservation->reservation_date)); ?></td>
                                    <td><?php echo date('H:i', strtotime($reservation->reservation_time)); ?></td>
                                    <td><?php echo $reservation->duration; ?> <?php _e('min', 'donyaye-game'); ?></td>
                                    <td>
                                        <span class="status <?php echo $reservation->status; ?>">
                                            <?php echo ucfirst($reservation->status); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($reservation->status === 'pending'): ?>
                                            <button class="dga-btn dga-btn-small dga-btn-secondary cancel-reservation" data-id="<?php echo $reservation->id; ?>">
                                                <?php _e('Cancel', 'donyaye-game'); ?>
                                            </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get wallet balance
     */
    private function get_wallet_balance($user_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_wallet_transactions';
        
        $credits = $wpdb->get_var($wpdb->prepare(
            "SELECT COALESCE(SUM(amount), 0) FROM $table_name 
            WHERE user_id = %d AND type = 'credit' AND status = 'completed'",
            $user_id
        ));
        
        $debits = $wpdb->get_var($wpdb->prepare(
            "SELECT COALESCE(SUM(amount), 0) FROM $table_name 
            WHERE user_id = %d AND type = 'debit' AND status = 'completed'",
            $user_id
        ));
        
        return floatval($credits) - floatval($debits);
    }
    
    /**
     * Get user transactions
     */
    private function get_user_transactions($user_id, $limit = 50) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_wallet_transactions';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name 
            WHERE user_id = %d 
            ORDER BY created_at DESC 
            LIMIT %d",
            $user_id,
            $limit
        ));
    }
    
    /**
     * Get user reservations
     */
    private function get_user_reservations($user_id, $limit = 50) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_reservations';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name 
            WHERE user_id = %d 
            ORDER BY reservation_date DESC, reservation_time DESC 
            LIMIT %d",
            $user_id,
            $limit
        ));
    }
} 