<?php
/**
 * Reservations Class
 */
class DGA_Reservations {
    
    public function __construct() {
        add_action('wp_ajax_dga_create_reservation', array($this, 'create_reservation'));
        add_action('wp_ajax_dga_cancel_reservation', array($this, 'cancel_reservation'));
        add_action('wp_ajax_dga_get_reservations', array($this, 'get_reservations_ajax'));
        add_action('wp_ajax_dga_get_available_slots', array($this, 'get_available_slots'));
    }
    
    /**
     * Create new reservation
     */
    public function create_reservation() {
        if (!wp_verify_nonce($_POST['nonce'], 'dga_nonce')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Security check failed.', 'donyaye-game')
            )));
        }
        
        if (!is_user_logged_in()) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('User not logged in.', 'donyaye-game')
            )));
        }
        
        $user_id = get_current_user_id();
        $service_id = sanitize_text_field($_POST['service_id']);
        $service_name = sanitize_text_field($_POST['service_name']);
        $reservation_date = sanitize_text_field($_POST['reservation_date']);
        $reservation_time = sanitize_text_field($_POST['reservation_time']);
        $duration = intval($_POST['duration']);
        $notes = sanitize_textarea_field($_POST['notes']);
        
        // Validate required fields
        if (empty($service_id) || empty($service_name) || empty($reservation_date) || empty($reservation_time)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('All required fields must be filled.', 'donyaye-game')
            )));
        }
        
        // Validate date (must be future date)
        $reservation_datetime = strtotime($reservation_date . ' ' . $reservation_time);
        if ($reservation_datetime <= time()) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Reservation must be for a future date and time.', 'donyaye-game')
            )));
        }
        
        // Check if slot is available
        if (!$this->is_slot_available($service_id, $reservation_date, $reservation_time, $duration)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Selected time slot is not available.', 'donyaye-game')
            )));
        }
        
        // Create reservation
        $reservation_id = $this->create_reservation_record($user_id, $service_id, $service_name, $reservation_date, $reservation_time, $duration, $notes);
        
        if (!$reservation_id) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Failed to create reservation.', 'donyaye-game')
            )));
        }
        
        wp_die(json_encode(array(
            'success' => true,
            'message' => __('Reservation created successfully!', 'donyaye-game'),
            'reservation_id' => $reservation_id
        )));
    }
    
    /**
     * Cancel reservation
     */
    public function cancel_reservation() {
        if (!wp_verify_nonce($_POST['nonce'], 'dga_nonce')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Security check failed.', 'donyaye-game')
            )));
        }
        
        if (!is_user_logged_in()) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('User not logged in.', 'donyaye-game')
            )));
        }
        
        $user_id = get_current_user_id();
        $reservation_id = intval($_POST['reservation_id']);
        
        // Get reservation
        $reservation = $this->get_reservation($reservation_id);
        
        if (!$reservation) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Reservation not found.', 'donyaye-game')
            )));
        }
        
        // Check if user owns this reservation
        if ($reservation->user_id != $user_id) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('You can only cancel your own reservations.', 'donyaye-game')
            )));
        }
        
        // Check if reservation can be cancelled
        if ($reservation->status !== 'pending' && $reservation->status !== 'confirmed') {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('This reservation cannot be cancelled.', 'donyaye-game')
            )));
        }
        
        // Cancel reservation
        if ($this->update_reservation_status($reservation_id, 'cancelled')) {
            wp_die(json_encode(array(
                'success' => true,
                'message' => __('Reservation cancelled successfully.', 'donyaye-game')
            )));
        } else {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Failed to cancel reservation.', 'donyaye-game')
            )));
        }
    }
    
    /**
     * Get reservations via AJAX
     */
    public function get_reservations_ajax() {
        if (!wp_verify_nonce($_POST['nonce'], 'dga_nonce')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Security check failed.', 'donyaye-game')
            )));
        }
        
        if (!is_user_logged_in()) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('User not logged in.', 'donyaye-game')
            )));
        }
        
        $user_id = get_current_user_id();
        $limit = isset($_POST['limit']) ? intval($_POST['limit']) : 50;
        $reservations = $this->get_user_reservations($user_id, $limit);
        
        wp_die(json_encode(array(
            'success' => true,
            'reservations' => $reservations
        )));
    }
    
    /**
     * Get available time slots
     */
    public function get_available_slots() {
        if (!wp_verify_nonce($_POST['nonce'], 'dga_nonce')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Security check failed.', 'donyaye-game')
            )));
        }
        
        $service_id = sanitize_text_field($_POST['service_id']);
        $date = sanitize_text_field($_POST['date']);
        
        if (empty($service_id) || empty($date)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Service ID and date are required.', 'donyaye-game')
            )));
        }
        
        $slots = $this->get_available_time_slots($service_id, $date);
        
        wp_die(json_encode(array(
            'success' => true,
            'slots' => $slots
        )));
    }
    
    /**
     * Create reservation record
     */
    private function create_reservation_record($user_id, $service_id, $service_name, $reservation_date, $reservation_time, $duration, $notes) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_reservations';
        
        $result = $wpdb->insert(
            $table_name,
            array(
                'user_id' => $user_id,
                'service_id' => $service_id,
                'service_name' => $service_name,
                'reservation_date' => $reservation_date,
                'reservation_time' => $reservation_time,
                'duration' => $duration,
                'notes' => $notes,
                'status' => 'pending'
            ),
            array('%d', '%s', '%s', '%s', '%s', '%d', '%s', '%s')
        );
        
        return $result ? $wpdb->insert_id : false;
    }
    
    /**
     * Check if time slot is available
     */
    private function is_slot_available($service_id, $date, $time, $duration) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_reservations';
        
        // Convert time to minutes for easier comparison
        $start_time = $this->time_to_minutes($time);
        $end_time = $start_time + $duration;
        
        // Check for overlapping reservations
        $overlapping = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table_name 
            WHERE service_id = %s 
            AND reservation_date = %s 
            AND status IN ('pending', 'confirmed')
            AND (
                (reservation_time <= %s AND reservation_time + duration > %s)
                OR (reservation_time < %d AND reservation_time + duration >= %d)
                OR (reservation_time >= %s AND reservation_time < %d)
            )",
            $service_id,
            $date,
            $start_time,
            $start_time,
            $start_time,
            $end_time,
            $start_time,
            $end_time
        ));
        
        return $overlapping == 0;
    }
    
    /**
     * Get available time slots
     */
    private function get_available_time_slots($service_id, $date) {
        // Define business hours (9 AM to 9 PM)
        $business_start = 9 * 60; // 9:00 AM in minutes
        $business_end = 21 * 60;  // 9:00 PM in minutes
        $slot_duration = 60; // 1 hour slots
        
        $available_slots = array();
        
        for ($time = $business_start; $time <= $business_end - $slot_duration; $time += $slot_duration) {
            if ($this->is_slot_available($service_id, $date, $this->minutes_to_time($time), $slot_duration)) {
                $available_slots[] = array(
                    'time' => $this->minutes_to_time($time),
                    'display_time' => date('H:i', strtotime($this->minutes_to_time($time)))
                );
            }
        }
        
        return $available_slots;
    }
    
    /**
     * Convert time string to minutes
     */
    private function time_to_minutes($time) {
        $parts = explode(':', $time);
        return intval($parts[0]) * 60 + intval($parts[1]);
    }
    
    /**
     * Convert minutes to time string
     */
    private function minutes_to_time($minutes) {
        $hours = floor($minutes / 60);
        $mins = $minutes % 60;
        return sprintf('%02d:%02d:00', $hours, $mins);
    }
    
    /**
     * Get reservation by ID
     */
    public function get_reservation($reservation_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_reservations';
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE id = %d",
            $reservation_id
        ));
    }
    
    /**
     * Update reservation status
     */
    private function update_reservation_status($reservation_id, $status) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_reservations';
        
        return $wpdb->update(
            $table_name,
            array('status' => $status),
            array('id' => $reservation_id),
            array('%s'),
            array('%d')
        );
    }
    
    /**
     * Get user reservations
     */
    public function get_user_reservations($user_id, $limit = 50) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_reservations';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name 
            WHERE user_id = %d 
            ORDER BY reservation_date DESC, reservation_time DESC 
            LIMIT %d",
            $user_id,
            $limit
        ));
    }
    
    /**
     * Get all reservations (admin function)
     */
    public function get_all_reservations($status = null, $limit = 100) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_reservations';
        
        $where_clause = '';
        $prepare_values = array($limit);
        
        if ($status) {
            $where_clause = 'WHERE status = %s';
            $prepare_values = array($status, $limit);
        }
        
        $sql = "SELECT * FROM $table_name $where_clause ORDER BY reservation_date DESC, reservation_time DESC LIMIT %d";
        
        return $wpdb->get_results($wpdb->prepare($sql, $prepare_values));
    }
    
    /**
     * Get reservations by date range
     */
    public function get_reservations_by_date_range($start_date, $end_date, $service_id = null) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_reservations';
        
        $where_clause = 'WHERE reservation_date BETWEEN %s AND %s';
        $prepare_values = array($start_date, $end_date);
        
        if ($service_id) {
            $where_clause .= ' AND service_id = %s';
            $prepare_values[] = $service_id;
        }
        
        $sql = "SELECT * FROM $table_name $where_clause ORDER BY reservation_date, reservation_time";
        
        return $wpdb->get_results($wpdb->prepare($sql, $prepare_values));
    }
    
    /**
     * Get upcoming reservations
     */
    public function get_upcoming_reservations($user_id = null, $days = 7) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_reservations';
        $future_date = date('Y-m-d', strtotime("+$days days"));
        
        $where_clause = 'WHERE reservation_date >= %s AND status IN ("pending", "confirmed")';
        $prepare_values = array(date('Y-m-d'), $future_date);
        
        if ($user_id) {
            $where_clause .= ' AND user_id = %d';
            $prepare_values[] = $user_id;
        }
        
        $sql = "SELECT * FROM $table_name $where_clause ORDER BY reservation_date, reservation_time";
        
        return $wpdb->get_results($wpdb->prepare($sql, $prepare_values));
    }
} 