<?php
/**
 * SMS Handler Class for SMS.ir Integration
 */
class DGA_SMS {
    
    private $api_key;
    private $template_id;
    private $api_url = 'https://api.sms.ir/v1/';
    
    public function __construct() {
        $this->api_key = get_option('dga_sms_api_key', '');
        $this->template_id = get_option('dga_sms_template_id', '');
    }
    
    /**
     * Send OTP SMS
     */
    public function send_otp($mobile) {
        // Generate OTP code
        $otp = $this->generate_otp();
        
        // Store OTP in database
        if (!$this->store_otp($mobile, $otp)) {
            return new WP_Error('otp_storage_failed', __('Failed to store OTP.', 'donyaye-game'));
        }
        
        // Send SMS via SMS.ir
        $result = $this->send_sms($mobile, $otp);
        
        if (is_wp_error($result)) {
            return $result;
        }
        
        return array(
            'success' => true,
            'message' => __('OTP sent successfully.', 'donyaye-game'),
            'mobile' => $mobile
        );
    }
    
    /**
     * Generate 6-digit OTP
     */
    private function generate_otp() {
        return str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
    }
    
    /**
     * Store OTP in database
     */
    private function store_otp($mobile, $otp) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_otp_codes';
        $expiry_time = date('Y-m-d H:i:s', time() + get_option('dga_otp_expiry', 300));
        
        $result = $wpdb->insert(
            $table_name,
            array(
                'mobile' => $mobile,
                'otp_code' => $otp,
                'expires_at' => $expiry_time,
                'used' => 0
            ),
            array('%s', '%s', '%s', '%d')
        );
        
        return $result !== false;
    }
    
    /**
     * Send SMS via SMS.ir API
     */
    private function send_sms($mobile, $otp) {
        if (empty($this->api_key)) {
            return new WP_Error('sms_config_missing', __('SMS configuration is missing.', 'donyaye-game'));
        }
        
        // If template ID is set, try template SMS first
        if (!empty($this->template_id)) {
            $sms_data = array(
                'Mobile' => $mobile,
                'TemplateId' => $this->template_id,
                'Parameters' => array(
                    array(
                        'Name' => 'Code',
                        'Value' => $otp
                    )
                )
            );
            
            $response = $this->send_template_sms($sms_data);
            
            if (!is_wp_error($response)) {
                return true;
            }
        }
        
        // Fallback to simple SMS
        $message = sprintf('کد تایید شما: %s', $otp);
        return $this->send_simple_sms($mobile, $message);
    }
    
    /**
     * Send template SMS via SMS.ir
     */
    private function send_template_sms($data) {
        $url = $this->api_url . 'send/verify';
        
        // Prepare data according to new SMS.ir API structure
        $request_data = array(
            'mobile' => $data['Mobile'],
            'templateId' => intval($data['TemplateId']),
            'parameters' => array(
                array(
                    'name' => $data['Parameters'][0]['Name'],
                    'value' => $data['Parameters'][0]['Value']
                )
            )
        );
        
        $headers = array(
            'x-api-key' => $this->api_key,
            'Content-Type' => 'application/json',
            'Accept' => 'text/plain'
        );
        
        $body = json_encode($request_data);
        
        // Log request for debugging if debug mode is enabled
        if (get_option('dga_sms_debug', 0)) {
            error_log('SMS.ir Template Request: ' . $url);
            error_log('SMS.ir Headers: ' . json_encode($headers));
            error_log('SMS.ir Body: ' . $body);
        }
        
        $response = wp_remote_post($url, array(
            'headers' => $headers,
            'body' => $body,
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            if (get_option('dga_sms_debug', 0)) {
                error_log('SMS.ir WP Error: ' . $response->get_error_message());
            }
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_headers = wp_remote_retrieve_headers($response);
        
        // Log response for debugging if debug mode is enabled
        if (get_option('dga_sms_debug', 0)) {
            error_log('SMS.ir Response Code: ' . $response_code);
            error_log('SMS.ir Response Headers: ' . json_encode($response_headers));
            error_log('SMS.ir Response Body: ' . $response_body);
        }
        
        if ($response_code !== 200) {
            return new WP_Error('sms_api_error', sprintf(__('SMS API error (HTTP %d): %s', 'donyaye-game'), $response_code, $response_body));
        }
        
        $result = json_decode($response_body, true);
        
        if (!$result) {
            return new WP_Error('sms_invalid_response', sprintf(__('Invalid JSON response from SMS API: %s', 'donyaye-game'), $response_body));
        }
        
        // Check for different possible response structures
        if (isset($result['isSuccessful'])) {
            if (!$result['isSuccessful']) {
                $message = isset($result['message']) ? $result['message'] : __('Unknown SMS API error.', 'donyaye-game');
                return new WP_Error('sms_failed', $message);
            }
        } elseif (isset($result['IsSuccessful'])) {
            if (!$result['IsSuccessful']) {
                $message = isset($result['Message']) ? $result['Message'] : __('Unknown SMS API error.', 'donyaye-game');
                return new WP_Error('sms_failed', $message);
            }
        } elseif (isset($result['success'])) {
            if (!$result['success']) {
                $message = isset($result['message']) ? $result['message'] : __('Unknown SMS API error.', 'donyaye-game');
                return new WP_Error('sms_failed', $message);
            }
        } else {
            // If none of the expected fields exist, log the full response
            if (get_option('dga_sms_debug', 0)) {
                error_log('SMS.ir Unexpected Response Structure: ' . json_encode($result));
            }
            return new WP_Error('sms_invalid_response', sprintf(__('Unexpected response structure from SMS API: %s', 'donyaye-game'), $response_body));
        }
        
        return true;
    }
    
    /**
     * Send simple SMS (fallback method)
     */
    private function send_simple_sms($mobile, $message) {
        $url = $this->api_url . 'send/bulk';
        
        $sms_data = array(
            'messages' => array($message),
            'mobileNumbers' => array($mobile),
            'lineNumber' => '', // Optional: Your dedicated line number
            'sendDateTime' => '', // Optional: Schedule sending
            'checkId' => '' // Optional: Check ID for tracking
        );
        
        $headers = array(
            'x-sms-ir-secure-token' => $this->api_key,
            'Content-Type' => 'application/json',
            'Accept' => 'text/plain'
        );
        
        $body = json_encode($sms_data);
        
        // Log request for debugging if debug mode is enabled
        if (get_option('dga_sms_debug', 0)) {
            error_log('SMS.ir Simple SMS Request: ' . $url);
            error_log('SMS.ir Simple SMS Headers: ' . json_encode($headers));
            error_log('SMS.ir Simple SMS Body: ' . $body);
        }
        
        $response = wp_remote_post($url, array(
            'headers' => $headers,
            'body' => $body,
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            if (get_option('dga_sms_debug', 0)) {
                error_log('SMS.ir Simple SMS WP Error: ' . $response->get_error_message());
            }
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_headers = wp_remote_retrieve_headers($response);
        
        // Log response for debugging if debug mode is enabled
        if (get_option('dga_sms_debug', 0)) {
            error_log('SMS.ir Simple SMS Response Code: ' . $response_code);
            error_log('SMS.ir Simple SMS Response Headers: ' . json_encode($response_headers));
            error_log('SMS.ir Simple SMS Response Body: ' . $response_body);
        }
        
        if ($response_code !== 200) {
            return new WP_Error('sms_api_error', sprintf(__('SMS API error (HTTP %d): %s', 'donyaye-game'), $response_code, $response_body));
        }
        
        $result = json_decode($response_body, true);
        
        if (!$result) {
            return new WP_Error('sms_invalid_response', sprintf(__('Invalid JSON response from SMS API: %s', 'donyaye-game'), $response_body));
        }
        
        // Check for different possible response structures
        if (isset($result['isSuccessful'])) {
            if (!$result['isSuccessful']) {
                $message = isset($result['message']) ? $result['message'] : __('Unknown SMS API error.', 'donyaye-game');
                return new WP_Error('sms_failed', $message);
            }
        } elseif (isset($result['IsSuccessful'])) {
            if (!$result['IsSuccessful']) {
                $message = isset($result['Message']) ? $result['Message'] : __('Unknown SMS API error.', 'donyaye-game');
                return new WP_Error('sms_failed', $message);
            }
        } elseif (isset($result['success'])) {
            if (!$result['success']) {
                $message = isset($result['message']) ? $result['message'] : __('Unknown SMS API error.', 'donyaye-game');
                return new WP_Error('sms_failed', $message);
            }
        } else {
            // If none of the expected fields exist, log the full response
            if (get_option('dga_sms_debug', 0)) {
                error_log('SMS.ir Simple SMS Unexpected Response Structure: ' . json_encode($result));
            }
            return new WP_Error('sms_invalid_response', sprintf(__('Unexpected response structure from SMS API: %s', 'donyaye-game'), $response_body));
        }
        
        return true;
    }
    
    /**
     * Test SMS configuration
     */
    public function test_sms_config() {
        if (empty($this->api_key)) {
            return new WP_Error('api_key_missing', __('API key is missing.', 'donyaye-game'));
        }
        
        // Try to get credit balance as a test
        return $this->get_credit_balance();
    }
    
    /**
     * Get SMS credit balance
     */
    public function get_credit_balance() {
        if (empty($this->api_key)) {
            return new WP_Error('api_key_missing', __('API key is missing.', 'donyaye-game'));
        }
        
        $url = $this->api_url . 'account/credit';
        
        $headers = array(
            'x-sms-ir-secure-token' => $this->api_key,
            'Accept' => 'text/plain'
        );
        
        // Log request for debugging if debug mode is enabled
        if (get_option('dga_sms_debug', 0)) {
            error_log('SMS.ir Credit Request: ' . $url);
            error_log('SMS.ir Credit Headers: ' . json_encode($headers));
        }
        
        $response = wp_remote_get($url, array(
            'headers' => $headers,
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            if (get_option('dga_sms_debug', 0)) {
                error_log('SMS.ir Credit WP Error: ' . $response->get_error_message());
            }
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_headers = wp_remote_retrieve_headers($response);
        
        // Log response for debugging if debug mode is enabled
        if (get_option('dga_sms_debug', 0)) {
            error_log('SMS.ir Credit Response Code: ' . $response_code);
            error_log('SMS.ir Credit Response Headers: ' . json_encode($response_headers));
            error_log('SMS.ir Credit Response Body: ' . $response_body);
        }
        
        if ($response_code !== 200) {
            return new WP_Error('api_error', sprintf(__('API error (HTTP %d): %s', 'donyaye-game'), $response_code, $response_body));
        }
        
        $result = json_decode($response_body, true);
        
        if (!$result) {
            return new WP_Error('invalid_response', sprintf(__('Invalid JSON response from API: %s', 'donyaye-game'), $response_body));
        }
        
        // Check for different possible response structures
        if (isset($result['isSuccessful'])) {
            if (!$result['isSuccessful']) {
                $message = isset($result['message']) ? $result['message'] : __('Unknown API error.', 'donyaye-game');
                return new WP_Error('api_failed', $message);
            }
            return isset($result['credit']) ? $result['credit'] : __('Unknown', 'donyaye-game');
        } elseif (isset($result['IsSuccessful'])) {
            if (!$result['IsSuccessful']) {
                $message = isset($result['Message']) ? $result['Message'] : __('Unknown API error.', 'donyaye-game');
                return new WP_Error('api_failed', $message);
            }
            return isset($result['Credit']) ? $result['Credit'] : __('Unknown', 'donyaye-game');
        } elseif (isset($result['success'])) {
            if (!$result['success']) {
                $message = isset($result['message']) ? $result['message'] : __('Unknown API error.', 'donyaye-game');
                return new WP_Error('api_failed', $message);
            }
            return isset($result['credit']) ? $result['credit'] : __('Unknown', 'donyaye-game');
        } else {
            // If none of the expected fields exist, log the full response
            if (get_option('dga_sms_debug', 0)) {
                error_log('SMS.ir Credit Unexpected Response Structure: ' . json_encode($result));
            }
            return new WP_Error('invalid_response', sprintf(__('Unexpected response structure from API: %s', 'donyaye-game'), $response_body));
        }
    }
    
    /**
     * Validate mobile number format (Iranian mobile)
     */
    public function validate_mobile($mobile) {
        // Remove any non-digit characters
        $mobile = preg_replace('/[^0-9]/', '', $mobile);
        
        // Check if it's a valid Iranian mobile number
        if (preg_match('/^09[0-9]{9}$/', $mobile)) {
            return $mobile;
        }
        
        return false;
    }
    
    /**
     * Clean expired OTP codes
     */
    public function clean_expired_otps() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_otp_codes';
        
        $result = $wpdb->query(
            "DELETE FROM $table_name WHERE expires_at < NOW()"
        );
        
        return $result;
    }
} 