<?php
/**
 * Wallet Class
 */
class DGA_Wallet {
    
    public function __construct() {
        add_action('wp_ajax_dga_charge_wallet', array($this, 'charge_wallet'));
        add_action('wp_ajax_dga_get_wallet_balance', array($this, 'get_wallet_balance_ajax'));
        add_action('wp_ajax_dga_get_transactions', array($this, 'get_transactions_ajax'));
    }
    
    /**
     * Charge wallet
     */
    public function charge_wallet() {
        if (!wp_verify_nonce($_POST['nonce'], 'dga_nonce')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Security check failed.', 'donyaye-game')
            )));
        }
        
        if (!is_user_logged_in()) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('User not logged in.', 'donyaye-game')
            )));
        }
        
        $user_id = get_current_user_id();
        $amount = floatval($_POST['amount']);
        $payment_method = sanitize_text_field($_POST['payment_method']);
        
        if ($amount < 1000) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Minimum amount is 1,000.', 'donyaye-game')
            )));
        }
        
        if (empty($payment_method)) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Please select a payment method.', 'donyaye-game')
            )));
        }
        
        // Create transaction record
        $transaction_id = $this->create_transaction($user_id, $amount, 'credit', 'Wallet charge', $payment_method);
        
        if (!$transaction_id) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Failed to create transaction.', 'donyaye-game')
            )));
        }
        
        // Process payment based on method
        $payment_result = $this->process_payment($transaction_id, $amount, $payment_method);
        
        if (is_wp_error($payment_result)) {
            // Update transaction status to failed
            $this->update_transaction_status($transaction_id, 'failed');
            
            wp_die(json_encode(array(
                'success' => false,
                'message' => $payment_result->get_error_message()
            )));
        }
        
        // Update transaction status to completed
        $this->update_transaction_status($transaction_id, 'completed');
        
        wp_die(json_encode(array(
            'success' => true,
            'message' => __('Wallet charged successfully!', 'donyaye-game'),
            'transaction_id' => $transaction_id,
            'new_balance' => $this->get_wallet_balance($user_id)
        )));
    }
    
    /**
     * Get wallet balance via AJAX
     */
    public function get_wallet_balance_ajax() {
        if (!wp_verify_nonce($_POST['nonce'], 'dga_nonce')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Security check failed.', 'donyaye-game')
            )));
        }
        
        if (!is_user_logged_in()) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('User not logged in.', 'donyaye-game')
            )));
        }
        
        $user_id = get_current_user_id();
        $balance = $this->get_wallet_balance($user_id);
        
        wp_die(json_encode(array(
            'success' => true,
            'balance' => $balance
        )));
    }
    
    /**
     * Get transactions via AJAX
     */
    public function get_transactions_ajax() {
        if (!wp_verify_nonce($_POST['nonce'], 'dga_nonce')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('Security check failed.', 'donyaye-game')
            )));
        }
        
        if (!is_user_logged_in()) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => __('User not logged in.', 'donyaye-game')
            )));
        }
        
        $user_id = get_current_user_id();
        $limit = isset($_POST['limit']) ? intval($_POST['limit']) : 50;
        $transactions = $this->get_user_transactions($user_id, $limit);
        
        wp_die(json_encode(array(
            'success' => true,
            'transactions' => $transactions
        )));
    }
    
    /**
     * Create transaction record
     */
    private function create_transaction($user_id, $amount, $type, $description, $reference_id = '') {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_wallet_transactions';
        
        $result = $wpdb->insert(
            $table_name,
            array(
                'user_id' => $user_id,
                'amount' => $amount,
                'type' => $type,
                'description' => $description,
                'reference_id' => $reference_id,
                'status' => 'pending'
            ),
            array('%d', '%f', '%s', '%s', '%s', '%s')
        );
        
        return $result ? $wpdb->insert_id : false;
    }
    
    /**
     * Update transaction status
     */
    private function update_transaction_status($transaction_id, $status) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_wallet_transactions';
        
        return $wpdb->update(
            $table_name,
            array('status' => $status),
            array('id' => $transaction_id),
            array('%s'),
            array('%d')
        );
    }
    
    /**
     * Process payment
     */
    private function process_payment($transaction_id, $amount, $payment_method) {
        switch ($payment_method) {
            case 'online':
                return $this->process_online_payment($transaction_id, $amount);
            case 'bank_transfer':
                return $this->process_bank_transfer($transaction_id, $amount);
            default:
                return new WP_Error('invalid_payment_method', __('Invalid payment method.', 'donyaye-game'));
        }
    }
    
    /**
     * Process online payment
     */
    private function process_online_payment($transaction_id, $amount) {
        // Here you would integrate with your payment gateway
        // For now, we'll simulate a successful payment
        
        // In a real implementation, you would:
        // 1. Redirect to payment gateway
        // 2. Handle payment callback
        // 3. Verify payment status
        // 4. Update transaction accordingly
        
        return true; // Simulate success
    }
    
    /**
     * Process bank transfer
     */
    private function process_bank_transfer($transaction_id, $amount) {
        // For bank transfers, the transaction remains pending
        // until manually confirmed by admin
        
        // You could send an email notification to admin here
        
        return true;
    }
    
    /**
     * Get wallet balance
     */
    public function get_wallet_balance($user_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_wallet_transactions';
        
        $credits = $wpdb->get_var($wpdb->prepare(
            "SELECT COALESCE(SUM(amount), 0) FROM $table_name 
            WHERE user_id = %d AND type = 'credit' AND status = 'completed'",
            $user_id
        ));
        
        $debits = $wpdb->get_var($wpdb->prepare(
            "SELECT COALESCE(SUM(amount), 0) FROM $table_name 
            WHERE user_id = %d AND type = 'debit' AND status = 'completed'",
            $user_id
        ));
        
        return floatval($credits) - floatval($debits);
    }
    
    /**
     * Get user transactions
     */
    public function get_user_transactions($user_id, $limit = 50) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_wallet_transactions';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name 
            WHERE user_id = %d 
            ORDER BY created_at DESC 
            LIMIT %d",
            $user_id,
            $limit
        ));
    }
    
    /**
     * Deduct from wallet
     */
    public function deduct_from_wallet($user_id, $amount, $description, $reference_id = '') {
        $current_balance = $this->get_wallet_balance($user_id);
        
        if ($current_balance < $amount) {
            return new WP_Error('insufficient_balance', __('Insufficient wallet balance.', 'donyaye-game'));
        }
        
        $transaction_id = $this->create_transaction($user_id, $amount, 'debit', $description, $reference_id);
        
        if (!$transaction_id) {
            return new WP_Error('transaction_failed', __('Failed to create debit transaction.', 'donyaye-game'));
        }
        
        // Mark as completed immediately for debits
        $this->update_transaction_status($transaction_id, 'completed');
        
        return $transaction_id;
    }
    
    /**
     * Add to wallet
     */
    public function add_to_wallet($user_id, $amount, $description, $reference_id = '') {
        $transaction_id = $this->create_transaction($user_id, $amount, 'credit', $description, $reference_id);
        
        if (!$transaction_id) {
            return new WP_Error('transaction_failed', __('Failed to create credit transaction.', 'donyaye-game'));
        }
        
        // Mark as completed immediately for credits
        $this->update_transaction_status($transaction_id, 'completed');
        
        return $transaction_id;
    }
    
    /**
     * Get transaction by ID
     */
    public function get_transaction($transaction_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_wallet_transactions';
        
        return $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE id = %d",
            $transaction_id
        ));
    }
    
    /**
     * Get pending transactions
     */
    public function get_pending_transactions($user_id = null) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'dga_wallet_transactions';
        
        if ($user_id) {
            return $wpdb->get_results($wpdb->prepare(
                "SELECT * FROM $table_name 
                WHERE user_id = %d AND status = 'pending' 
                ORDER BY created_at DESC",
                $user_id
            ));
        } else {
            return $wpdb->get_results(
                "SELECT * FROM $table_name 
                WHERE status = 'pending' 
                ORDER BY created_at DESC"
            );
        }
    }
} 